unit FluxMatrix;
//---------------------------------------------------------------------------
// FluxMatrix.pas                                       Modified: 19-Oct-2006
// High-level implementation of 4x4 matrix w/32-byte alignment    Version 1.0
//---------------------------------------------------------------------------
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is FluxMatrix.pas.
//
// The Initial Developer of the Original Code is M. Sc. Yuriy Kotsarenko.
// Portions created by M. Sc. Yuriy Kotsarenko are Copyright (C) 2007,
// M. Sc. Yuriy Kotsarenko. All Rights Reserved.
//---------------------------------------------------------------------------
interface

//---------------------------------------------------------------------------
uses
 Vectors3, Vectors4, Matrices4;

//---------------------------------------------------------------------------
type
 TFluxMatrix = class
 private
  MemAddr: Pointer;
  FRawMtx: PMatrix4;

 public
  property RawMtx: PMatrix4 read FRawMtx;

  procedure LoadIdentity();
  procedure LoadZero();

  procedure Translate(dx, dy, dz: Real); overload;
  procedure Translate(const v: TVector3); overload;
  procedure Translate(const v: TVector4); overload;

  procedure Scale(const v: TVector3); overload;
  procedure Scale(dx, dy, dz: Real); overload;
  procedure RotateX(Phi: Real);
  procedure RotateY(Phi: Real);
  procedure RotateZ(Phi: Real);

  procedure RotateXLocal(Phi: Real);
  procedure RotateYLocal(Phi: Real);
  procedure RotateZLocal(Phi: Real);

  procedure Multiply(const SrcMtx: TMatrix4); overload;
  procedure Multiply(Source: TFluxMatrix); overload;

  procedure LoadRotation(SrcMtx: PMatrix4);

  procedure LookAt(const Origin, Target, Roof: TVector4);

  procedure PerspectiveFOVY(FieldOfView, AspectRatio, MinRange, MaxRange: Real);
  procedure PerspectiveFOVX(FieldOfView, AspectRatio, MinRange, MaxRange: Real);
  procedure PerspectiveVOL(Width, Height, MinRange, MaxRange: Real);
  procedure PerspectiveBDS(Left, Right, Top, Bottom, MinRange, MaxRange: Real);
  procedure OrthogonalVOL(Width, Height, MinRange, MaxRange: Real);
  procedure OrthogonalBDS(Left, Right, Top, Bottom, MinRange, MaxRange: Real);

  constructor Create();
  destructor Destroy(); override;
 end;

//---------------------------------------------------------------------------
implementation

//---------------------------------------------------------------------------
const
 Grad2Rad = Pi / 180.0;

//---------------------------------------------------------------------------
constructor TFluxMatrix.Create();
begin
 inherited;

 MemAddr:= AllocMem(SizeOf(TMatrix4) + 16);
 FRawMtx:= Pointer(Integer(MemAddr) + ($10 - (Integer(MemAddr) and $0F)));

 LoadIdentity();
end;

//---------------------------------------------------------------------------
destructor TFluxMatrix.Destroy();
begin
 FreeMem(MemAddr);

 inherited;
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.LoadIdentity();
begin
 Move(IdentityMtx4, FRawMtx^, SizeOf(TMatrix4));
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.LoadZero();
begin
 FillChar(FRawMtx^, SizeOf(TMatrix4), 0);
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.Translate(dx, dy, dz: Real);
begin
 FRawMtx^:= FRawMtx^ * TranslateMtx4(Vector3(dx, dy, dz));
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.Translate(const v: TVector3);
begin
 FRawMtx^:= FRawMtx^ * TranslateMtx4(v);
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.Translate(const v: TVector4);
begin
 Translate(v.x, v.y, v.z);
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.RotateX(Phi: Real);
begin
 FRawMtx^:= FRawMtx^ * RotateXMtx4(Phi);
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.RotateY(Phi: Real);
begin
 FRawMtx^:= FRawMtx^ * RotateYMtx4(Phi);
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.RotateZ(Phi: Real);
begin
 FRawMtx^:= FRawMtx^ * RotateZMtx4(Phi);
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.Scale(const v: TVector3);
begin
 FRawMtx^:= FRawMtx^ * ScaleMtx4(v);
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.Scale(dx, dy, dz: Real);
begin
 FRawMtx^:= FRawMtx^ * ScaleMtx4(Vector3(dx, dy, dz));
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.RotateXLocal(Phi: Real);
var
 Axis: TVector4;
begin
 Axis.x:= FRawMtx.Data[0, 0];
 Axis.y:= FRawMtx.Data[0, 1];
 Axis.z:= FRawMtx.Data[0, 2];
 Axis.w:= 1.0;

 FRawMtx^:= FRawMtx^ * RotateMtx4(Axis, Phi);
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.RotateYLocal(Phi: Real);
var
 Axis: TVector4;
begin
 Axis.x:= FRawMtx.Data[1, 0];
 Axis.y:= FRawMtx.Data[1, 1];
 Axis.z:= FRawMtx.Data[1, 2];
 Axis.w:= 1.0;

 FRawMtx^:= FRawMtx^ * RotateMtx4(Axis, Phi);
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.RotateZLocal(Phi: Real);
var
 Axis: TVector4;
begin
 Axis.x:= FRawMtx.Data[2, 0];
 Axis.y:= FRawMtx.Data[2, 1];
 Axis.z:= FRawMtx.Data[2, 2];
 Axis.w:= 1.0;

 FRawMtx^:= FRawMtx^ * RotateMtx4(Axis, Phi);
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.Multiply(const SrcMtx: TMatrix4);
begin
 FRawMtx^:= FRawMtx^ * SrcMtx;
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.Multiply(Source: TFluxMatrix);
begin
 Multiply(Source.RawMtx^);
end;

//--------------------------------------------------------------------------
procedure TFluxMatrix.LookAt(const Origin, Target, Roof: TVector4);
begin
 FRawMtx^:= FRawMtx^ * LookAtMtx4(Origin, Target, Roof);
end;

//--------------------------------------------------------------------------
procedure TFluxMatrix.PerspectiveFOVY(FieldOfView, AspectRatio, MinRange,
 MaxRange: Real);
begin
 FRawMtx^:= PerspectiveFOVYMtx4(FieldOfView, AspectRatio, MinRange,
  MaxRange);
end;

//--------------------------------------------------------------------------
procedure TFluxMatrix.PerspectiveFOVX(FieldOfView, AspectRatio, MinRange,
 MaxRange: Real);
begin
 FRawMtx^:= PerspectiveFOVXMtx4(FieldOfView, AspectRatio, MinRange,
  MaxRange);
end;

//--------------------------------------------------------------------------
procedure TFluxMatrix.PerspectiveVOL(Width, Height, MinRange,
 MaxRange: Real);
begin
 FRawMtx^:= PerspectiveVOLMtx4(Width, Height, MinRange, MaxRange);
end;

//--------------------------------------------------------------------------
procedure TFluxMatrix.PerspectiveBDS(Left, Right, Top, Bottom, MinRange,
 MaxRange: Real);
begin
 FRawMtx^:= PerspectiveBDSMtx4(Left, Right, Top, Bottom, MinRange, MaxRange);
end;

//--------------------------------------------------------------------------
procedure TFluxMatrix.OrthogonalVOL(Width, Height, MinRange, MaxRange: Real);
begin
 FRawMtx^:= OrthogonalVOLMtx4(Width, Height, MinRange, MaxRange);
end;

//--------------------------------------------------------------------------
procedure TFluxMatrix.OrthogonalBDS(Left, Right, Top, Bottom, MinRange,
 MaxRange: Real);
begin
 FRawMtx^:= OrthogonalBDSMtx4(Left, Right, Top, Bottom, MinRange, MaxRange);
end;

//---------------------------------------------------------------------------
procedure TFluxMatrix.LoadRotation(SrcMtx: PMatrix4);
var
 i: Integer;
begin
 Move(SrcMtx^, FRawMtx^, SizeOf(TMatrix4));

 for i:= 0 to 3 do
  begin
   FRawMtx.Data[3, i]:= 0.0;
   FRawMtx.Data[i, 3]:= 0.0;
  end;
  
 FRawMtx.Data[3, 3]:= 1.0; 
end;

//---------------------------------------------------------------------------
end.
