unit FluxMeshes4;
//---------------------------------------------------------------------------
// FluxModels.pas                                       Modified: 05-Oct-2006
// The definition of 3D model of FluxScene                        Version 1.0
//---------------------------------------------------------------------------
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is FluxFaces.pas.
//
// The Initial Developer of the Original Code is M. Sc. Yuriy Kotsarenko.
// Portions created by M. Sc. Yuriy Kotsarenko are Copyright (C) 2007,
// M. Sc. Yuriy Kotsarenko. All Rights Reserved.
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
interface

//---------------------------------------------------------------------------
uses
 Classes, SysUtils, Vectors2, Vectors4, FluxMeshFaces, FluxMeshes;

//---------------------------------------------------------------------------
type
 TFluxMesh4 = class
 private
  FVertices   : TVectors4;
  FNormals    : TVectors4;
  FTexCoords  : TPoints2;
  FFaceNormals: TVectors4;
  FFaceMidPts : TVectors4;
  FFaces      : TFluxMeshFaces;
 public
  // primary components
  property Vertices : TVectors4 read FVertices;
  property Normals  : TVectors4 read FNormals;
  property TexCoords: TPoints2 read FTexCoords;

  // secondary components
  property FaceNormals: TVectors4 read FFaceNormals;
  property FaceMidPts : TVectors4 read FFaceMidPts;

  // mesh faces
  property Faces: TFluxMeshFaces read FFaces;

  procedure LoadFromMesh(Mesh: TFluxMesh);

  constructor Create();
  destructor Destroy(); override;
 end;

//---------------------------------------------------------------------------
 TFluxMeshes4 = class
 private
  Meshes: array of TFluxMesh4;

  function GetCount(): Integer;
  function GetItem(Num: Integer): TFluxMesh4;
 public
  property Count: Integer read GetCount;
  property Items[Num: Integer]: TFluxMesh4 read GetItem; default;

  function NewMesh(): Integer;

  procedure Remove(Num: Integer);
  procedure RemoveAll();

  constructor Create();
  destructor Destroy(); override;
 end;

//---------------------------------------------------------------------------
implementation

//---------------------------------------------------------------------------
constructor TFluxMesh4.Create();
begin
 inherited Create();

 FVertices := TVectors4.Create();
 FNormals  := TVectors4.Create();
 FTexCoords:= TPoints2.Create();

 FFaceNormals:= TVectors4.Create();
 FFaceMidPts := TVectors4.Create();

 FFaces:= TFluxMeshFaces.Create();
end;

//---------------------------------------------------------------------------
destructor TFluxMesh4.Destroy();
begin
 FFaces.Free();
 FFaceMidPts.Free();
 FFaceNormals.Free();

 FTexCoords.Free();
 FNormals.Free();
 FVertices.Free();

 inherited;
end;

//---------------------------------------------------------------------------
procedure TFluxMesh4.LoadFromMesh(Mesh: TFluxMesh);
var
 i: Integer;
begin
 FVertices.RemoveAll();
 for i:= 0 to Mesh.Vertices.Count - 1 do
  FVertices.Add(TVector4(Mesh.Vertices[i]));

 FNormals.RemoveAll();
 for i:= 0 to Mesh.Normals.Count - 1 do
  FNormals.Add(TVector4(Mesh.Normals[i]));

 FTexCoords.RemoveAll();
 for i:= 0 to Mesh.TexCoords.Count - 1 do
  FTexCoords.Add(Mesh.TexCoords[i]^);

 FFaceNormals.RemoveAll();
 for i:= 0 to Mesh.FaceNormals.Count - 1 do
  FFaceNormals.Add(TVector4(Mesh.FaceNormals[i]));

 FFaceMidPts.RemoveAll();
 for i:= 0 to Mesh.FaceMidPts.Count - 1 do
  FFaceMidPts.Add(TVector4(Mesh.FaceMidPts[i]));

 FFaces.RemoveAll();
 for i:= 0 to Mesh.Faces.Count - 1 do
  FFaces.Insert(Mesh.Faces[i]);
end;

//---------------------------------------------------------------------------
constructor TFluxMeshes4.Create();
begin
 inherited;

 SetLength(Meshes, 0);
end;

//---------------------------------------------------------------------------
destructor TFluxMeshes4.Destroy();
begin
 RemoveAll();

 inherited;
end;

//---------------------------------------------------------------------------
function TFluxMeshes4.GetCount(): Integer;
begin
 Result:= Length(Meshes);
end;

//---------------------------------------------------------------------------
function TFluxMeshes4.GetItem(Num: Integer): TFluxMesh4;
begin
 if (Num >= 0)and(Num < Length(Meshes)) then Result:= Meshes[Num]
  else Result:= nil;
end;

//---------------------------------------------------------------------------
procedure TFluxMeshes4.Remove(Num: Integer);
var
 i: Integer;
begin
 if (Num < 0)or(Num >= Length(Meshes)) then Exit;

 Meshes[Num].Free();

 for i:= Num to Length(Meshes) - 2 do
  Meshes[i]:= Meshes[i + 1];

 SetLength(Meshes, Length(Meshes) - 1);
end;

//---------------------------------------------------------------------------
procedure TFluxMeshes4.RemoveAll();
var
 i: Integer;
begin
 for i:= 0 to Length(Meshes) - 1 do
  if (Meshes[i] <> nil) then
   begin
    Meshes[i].Free();
    Meshes[i]:= nil;
   end;

 SetLength(Meshes, 0);
end;

//---------------------------------------------------------------------------
function TFluxMeshes4.NewMesh(): Integer;
begin
 Result:= Length(Meshes);
 SetLength(Meshes, Result + 1);

 Meshes[Result]:= TFluxMesh4.Create();
end;

//---------------------------------------------------------------------------
end.
