unit FluxTorusKnot;
//---------------------------------------------------------------------------
// FluxTorusKnot.pas                                    Modified: 03-Dec-2007
// Torus PQ Knot implementation for FluxScene                     Version 1.0
//---------------------------------------------------------------------------
// This code is based on the description at:
//  http://www.blackpawn.com/texts/pqtorus/default.html
//---------------------------------------------------------------------------
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is FluxTorusKnot.pas.
//
// The Initial Developer of the Original Code is M. Sc. Yuriy Kotsarenko.
// Portions created by M. Sc. Yuriy Kotsarenko are Copyright (C) 2007,
// M. Sc. Yuriy Kotsarenko. All Rights Reserved.
//---------------------------------------------------------------------------
interface

//---------------------------------------------------------------------------
uses
 Windows, Math, Vectors2, Vectors3, FluxMeshes, FluxMeshes4;

//---------------------------------------------------------------------------
function MakeTorusPQ(Radius, TubeRadius: Single; p, q: Integer; Rings,
 Sides: Integer; TexTilesX, TexTilesY: Single): TFluxMesh4;

//---------------------------------------------------------------------------
implementation

//---------------------------------------------------------------------------
function MakeTorusPQ(Radius, TubeRadius: Single; p,
 q: Integer; Rings, Sides: Integer; TexTilesX, TexTilesY: Single): TFluxMesh4;
var
 Mesh: TFluxMesh;
 SideBlock: Integer;
 i, j, ni, nj: Integer;
 Phi, ThetaInc: Single;
 Theta, PhiInc, Phi1, r: Single;
 Vertex, Pos, Next, t, n, b: TVector3;
 Point: TPoint2;
 Normal: TVector3;
begin
 SideBlock:= Sides + 1;

 Mesh:= TFluxMesh.Create();

 Phi:= 0.0;
 PhiInc:= 2.0 * Pi / Rings;
 ThetaInc:= 2.0 * Pi / Sides;

 for j:= 0 to Rings do
  begin
   Theta:= 0.0;

   r:= 0.5 * (2.0 + Sin(q * Phi)) * Radius;
   Pos.x:= r * Cos(p * Phi);
   Pos.y:= r * Cos(q * Phi);
   Pos.z:= r * Sin(p * Phi);

   Phi1:= Phi + PhiInc;
   r:= 0.5 * (2.0 + Sin(q * Phi1)) * Radius;
   Next.x:= r * Cos(p * Phi1);
   Next.y:= r * Cos(q * Phi1);
   Next.z:= r * Sin(p * Phi1);

   t:= Next - Pos;
   n:= Next + Pos;
   b:= Norm3(Cross3(t, n));
   n:= Norm3(Cross3(b, t));

   for i:= 0 to Sides do
    begin
     Point.x:= Sin(Theta) * TubeRadius;
     Point.y:= Cos(Theta) * TubeRadius;

     Vertex:= Pos + (n * Point.x) + (b * Point.y);
     Normal:= Norm3(Vertex - Pos);

     Mesh.Vertices.Add(Vertex);
     Mesh.Normals.Add(Normal);
     Mesh.TexCoords.Add(Point2(Phi * TexTilesX / (2.0 * Pi),
      Theta * TexTilesY / (2.0 * Pi)));

     ni:= i + 1;
     nj:= j + 1;

     if (i < Sides)and(j < Rings) then
      begin
       Mesh.Faces.Insert(
        i + j * SideBlock,
        i + nj * SideBlock,
        ni + nj * SideBlock);

       Mesh.Faces.Insert(
        i + j * SideBlock,
        ni + nj * SideBlock,
        ni + j * SideBlock);
      end;

     Theta:= Theta + ThetaInc;
    end;

   Phi:= Phi + PhiInc;
  end;

 Mesh.FindFaceNormals();
 Mesh.FindFaceMidPoints();

 Result:= TFluxMesh4.Create();
 Result.LoadFromMesh(Mesh);

 Mesh.Free();
end;

//---------------------------------------------------------------------------
end.
