unit SuHoClipPerspective;
//---------------------------------------------------------------------------
// SuHoClip.pas                                         Modified: 03-Dec-2007
// Sutherland - Hodgman Polygon Clipping                          Version 1.0
//
// This is an implementation of Sutherland-Hodgman algorithm for clipping
// polygons with optional perspective correction.
//---------------------------------------------------------------------------
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is SuHoClip.pas.
//
// The Initial Developer of the Original Code is M. Sc. Yuriy Kotsarenko.
// Portions created by M. Sc. Yuriy Kotsarenko are Copyright (C) 2007,
// M. Sc. Yuriy Kotsarenko. All Rights Reserved.
//---------------------------------------------------------------------------
interface

//---------------------------------------------------------------------------
uses
 Vectors2, Vectors3;

//---------------------------------------------------------------------------
// The following directive enables perspective correction when clipping
// occurs. This is needed to avoid distortion when using perspective correct
// texture mapping.
//---------------------------------------------------------------------------
{$define PerspectiveClip}

//---------------------------------------------------------------------------
type
 PClipItem = ^TClipItem;
 TClipItem = record
  Position: TVector3;
  TexCoord: TPoint2;
  Visible : Boolean;
 end;

//---------------------------------------------------------------------------
// The following parameters define the clipping rectangle.
//---------------------------------------------------------------------------
var
 LeftClip  : Integer = 0;
 RightClip : Single  = 640;
 TopClip   : Integer = 0;
 BottomClip: Integer = 480;

//---------------------------------------------------------------------------
procedure ClipLeft(inVertices, outVertices: PClipItem; InCount: Integer;
 out OutCount: Integer);
procedure ClipRight(inVertices, outVertices: PClipItem; InCount: Integer;
 out OutCount: Integer);
procedure ClipTop(inVertices, outVertices: PClipItem; InCount: Integer;
 out OutCount: Integer);
procedure ClipBottom(inVertices, outVertices: PClipItem; InCount: Integer;
 out OutCount: Integer);

//---------------------------------------------------------------------------
implementation

//---------------------------------------------------------------------------
procedure ClipLeft(inVertices, outVertices: PClipItem; InCount: Integer;
 out OutCount: Integer);
var
 i: Integer;
 CurItem, NextItem, DestItem: PClipItem;
 Theta, nTheta: Single;
 {$ifdef PerspectiveClip}iz0, iz1, zm: Single;{$endif}
begin
 OutCount:= 0;

 CurItem:= inVertices;
 for i:= 0 to InCount - 1 do
  begin
   CurItem.Visible:= CurItem.Position.x >= LeftClip;
   Inc(CurItem);
  end;

 CurItem := inVertices;
 NextItem:= Pointer(Integer(inVertices) + SizeOf(TClipItem));
 DestItem:= outVertices;

 for i:= 0 to InCount - 1 do
  begin
   if (i = InCount - 1) then NextItem:= inVertices;

   if (CurItem.Visible) then
    begin
     Move(CurItem^, DestItem^, SizeOf(TClipItem));
     Inc(DestItem);
     Inc(OutCount);
    end;

   if ((CurItem.Visible <> NextItem.Visible)) then
    begin
     Theta:= (LeftClip - CurItem.Position.x) /
      (NextItem.Position.x - CurItem.Position.x);
     nTheta:= 1.0 - Theta;

     DestItem.Position.x:= LeftClip;
     DestItem.Position.y:= CurItem.Position.y * nTheta +
      NextItem.Position.y * Theta;

     {$ifdef PerspectiveClip}
     iz0:= 1.0 / CurItem.Position.z;
     iz1:= 1.0 / NextItem.Position.z;
     zm := iz0 * nTheta + iz1 * Theta;
     DestItem.Position.z:= 1.0 / zm;
     {$else}
     DestItem.Position.z:= CurItem.Position.z * nTheta +
      NextItem.Position.z * Theta;
     {$endif} 

     {$ifdef PerspectiveClip}
     DestItem.TexCoord.x:= ((CurItem.TexCoord.x / CurItem.Position.z) * nTheta +
      (NextItem.TexCoord.x / NextItem.Position.z) * Theta) *
      DestItem.Position.z;
     DestItem.TexCoord.y:= ((CurItem.TexCoord.y / CurItem.Position.z) * nTheta +
      (NextItem.TexCoord.y / NextItem.Position.z) * Theta) *
      DestItem.Position.z;
     {$else}
     DestItem.TexCoord.x:= CurItem.TexCoord.x * nTheta +
      NextItem.TexCoord.x * Theta;
     DestItem.TexCoord.y:= CurItem.TexCoord.y * nTheta +
      NextItem.TexCoord.y * Theta;
     {$endif}

     Inc(DestItem);
     Inc(OutCount);
    end;

   Inc(CurItem);
   Inc(NextItem); 
  end;
end;

//---------------------------------------------------------------------------
procedure ClipRight(inVertices, outVertices: PClipItem; InCount: Integer;
 out OutCount: Integer);
var
 i: Integer;
 CurItem, NextItem, DestItem: PClipItem;
 Theta, nTheta: Single;
 {$ifdef PerspectiveClip}iz0, iz1, zm: Single;{$endif}
begin
 OutCount:= 0;

 CurItem:= inVertices;
 for i:= 0 to InCount - 1 do
  begin
   CurItem.Visible:= CurItem.Position.x <= RightClip;
   Inc(CurItem);
  end;

 CurItem := inVertices;
 NextItem:= Pointer(Integer(inVertices) + SizeOf(TClipItem));
 DestItem:= outVertices;

 for i:= 0 to InCount - 1 do
  begin
   if (i = InCount - 1) then NextItem:= inVertices;

   if (CurItem.Visible) then
    begin
     Move(CurItem^, DestItem^, SizeOf(TClipItem));
     Inc(DestItem);
     Inc(OutCount);
    end;

   if ((CurItem.Visible <> NextItem.Visible)) then
    begin
     Theta:= (RightClip - CurItem.Position.x) /
      (NextItem.Position.x - CurItem.Position.x);
     nTheta:= 1.0 - Theta;

     DestItem.Position.x:= RightClip;
     DestItem.Position.y:= CurItem.Position.y * nTheta +
      NextItem.Position.y * Theta;

     {$ifdef PerspectiveClip}
     iz0:= 1.0 / CurItem.Position.z;
     iz1:= 1.0 / NextItem.Position.z;
     zm := iz0 * nTheta + iz1 * Theta;
     DestItem.Position.z:= 1.0 / zm;
     {$else}
     DestItem.Position.z:= CurItem.Position.z * nTheta +
      NextItem.Position.z * Theta;
     {$endif} 

     {$ifdef PerspectiveClip}
     DestItem.TexCoord.x:= ((CurItem.TexCoord.x / CurItem.Position.z) * nTheta +
      (NextItem.TexCoord.x / NextItem.Position.z) * Theta) *
      DestItem.Position.z;
     DestItem.TexCoord.y:= ((CurItem.TexCoord.y / CurItem.Position.z) * nTheta +
      (NextItem.TexCoord.y / NextItem.Position.z) * Theta) *
      DestItem.Position.z;
     {$else}
     DestItem.TexCoord.x:= CurItem.TexCoord.x * nTheta +
      NextItem.TexCoord.x * Theta;
     DestItem.TexCoord.y:= CurItem.TexCoord.y * nTheta +
      NextItem.TexCoord.y * Theta;
     {$endif}
      
     Inc(DestItem);
     Inc(OutCount);
    end;

   Inc(CurItem);
   Inc(NextItem); 
  end;
end;

//---------------------------------------------------------------------------
procedure ClipTop(inVertices, outVertices: PClipItem; InCount: Integer;
 out OutCount: Integer);
var
 i: Integer;
 CurItem, NextItem, DestItem: PClipItem;
 Theta, nTheta: Single;
 {$ifdef PerspectiveClip}iz0, iz1, zm: Single;{$endif}
begin
 OutCount:= 0;

 CurItem:= inVertices;
 for i:= 0 to InCount - 1 do
  begin
   CurItem.Visible:= CurItem.Position.y >= TopClip;
   Inc(CurItem);
  end;

 CurItem := inVertices;
 NextItem:= Pointer(Integer(inVertices) + SizeOf(TClipItem));
 DestItem:= outVertices;

 for i:= 0 to InCount - 1 do
  begin
   if (i = InCount - 1) then NextItem:= inVertices;

   if (CurItem.Visible) then
    begin
     Move(CurItem^, DestItem^, SizeOf(TClipItem));
     Inc(DestItem);
     Inc(OutCount);
    end;

   if ((CurItem.Visible <> NextItem.Visible)) then
    begin
     Theta:= (TopClip - CurItem.Position.y) /
      (NextItem.Position.y - CurItem.Position.y);
     nTheta:= 1.0 - Theta;

     DestItem.Position.y:= TopClip;
     DestItem.Position.x:= CurItem.Position.x * nTheta +
      NextItem.Position.x * Theta;

     {$ifdef PerspectiveClip}
     iz0:= 1.0 / CurItem.Position.z;
     iz1:= 1.0 / NextItem.Position.z;
     zm := iz0 * nTheta + iz1 * Theta;
     DestItem.Position.z:= 1.0 / zm;
     {$else}
     DestItem.Position.z:= CurItem.Position.z * nTheta +
      NextItem.Position.z * Theta;
     {$endif}

     {$ifdef PerspectiveClip}
     DestItem.TexCoord.x:= ((CurItem.TexCoord.x / CurItem.Position.z) * nTheta +
      (NextItem.TexCoord.x / NextItem.Position.z) * Theta) *
      DestItem.Position.z;
     DestItem.TexCoord.y:= ((CurItem.TexCoord.y / CurItem.Position.z) * nTheta +
      (NextItem.TexCoord.y / NextItem.Position.z) * Theta) *
      DestItem.Position.z;
     {$else}
     DestItem.TexCoord.x:= CurItem.TexCoord.x * nTheta +
      NextItem.TexCoord.x * Theta;
     DestItem.TexCoord.y:= CurItem.TexCoord.y * nTheta +
      NextItem.TexCoord.y * Theta;
     {$endif}

     Inc(DestItem);
     Inc(OutCount);
    end;

   Inc(CurItem);
   Inc(NextItem);
  end;
end;

//---------------------------------------------------------------------------
procedure ClipBottom(inVertices, outVertices: PClipItem; InCount: Integer;
 out OutCount: Integer);
var
 i: Integer;
 CurItem, NextItem, DestItem: PClipItem;
 Theta, nTheta: Single;
 {$ifdef PerspectiveClip}iz0, iz1, zm: Single;{$endif}
begin
 OutCount:= 0;

 CurItem:= inVertices;
 for i:= 0 to InCount - 1 do
  begin
   CurItem.Visible:= CurItem.Position.y <= BottomClip;
   Inc(CurItem);
  end;

 CurItem := inVertices;
 NextItem:= Pointer(Integer(inVertices) + SizeOf(TClipItem));
 DestItem:= outVertices;

 for i:= 0 to InCount - 1 do
  begin
   if (i = InCount - 1) then NextItem:= inVertices;

   if (CurItem.Visible) then
    begin
     Move(CurItem^, DestItem^, SizeOf(TClipItem));
     Inc(DestItem);
     Inc(OutCount);
    end;

   if ((CurItem.Visible <> NextItem.Visible)) then
    begin
     Theta:= (BottomClip - CurItem.Position.y) /
      (NextItem.Position.y - CurItem.Position.y);
     nTheta:= 1.0 - Theta;

     DestItem.Position.y:= BottomClip;
     DestItem.Position.x:= CurItem.Position.x * nTheta +
      NextItem.Position.x * Theta;

     {$ifdef PerspectiveClip}
     iz0:= 1.0 / CurItem.Position.z;
     iz1:= 1.0 / NextItem.Position.z;
     zm := iz0 * nTheta + iz1 * Theta;
     DestItem.Position.z:= 1.0 / zm;
     {$else}
     DestItem.Position.z:= CurItem.Position.z * nTheta +
      NextItem.Position.z * Theta;
     {$endif}

     {$ifdef PerspectiveClip}
     DestItem.TexCoord.x:= ((CurItem.TexCoord.x / CurItem.Position.z) * nTheta +
      (NextItem.TexCoord.x / NextItem.Position.z) * Theta) *
      DestItem.Position.z;
     DestItem.TexCoord.y:= ((CurItem.TexCoord.y / CurItem.Position.z) * nTheta +
      (NextItem.TexCoord.y / NextItem.Position.z) * Theta) *
      DestItem.Position.z;
     {$else}
     DestItem.TexCoord.x:= CurItem.TexCoord.x * nTheta +
      NextItem.TexCoord.x * Theta;
     DestItem.TexCoord.y:= CurItem.TexCoord.y * nTheta +
      NextItem.TexCoord.y * Theta;
     {$endif}

     Inc(DestItem);
     Inc(OutCount);
    end;

   Inc(CurItem);
   Inc(NextItem);
  end;
end;

//---------------------------------------------------------------------------
end.

