unit FixedColorDef;
//---------------------------------------------------------------------------
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is FixedColorDef.pas.
//
// The Initial Developer of the Original Code is M. Sc. Yuriy Kotsarenko.
// Portions created by M. Sc. Yuriy Kotsarenko are Copyright (C) 2007,
// M. Sc. Yuriy Kotsarenko. All Rights Reserved.
//---------------------------------------------------------------------------
interface

//---------------------------------------------------------------------------
uses
 FluxUtils;

//---------------------------------------------------------------------------
type
 TFixedColor = record
  r, g, b: Integer;

  class operator Add(const a, b: TFixedColor): TFixedColor;
  class operator Subtract(const a, b: TFixedColor): TFixedColor;

  class operator Implicit(const c: TFixedColor): Cardinal;
  class operator Implicit(c: Cardinal): TFixedColor;

  procedure Clamp();
 end;

//---------------------------------------------------------------------------
function MinColor(const c1, c2, c3: TFixedColor): TFixedColor;
function MaxColor(const c1, c2, c3: TFixedColor): TFixedColor;
function ClampColor(const Color, MinCol, MaxCol: TFixedColor): TFixedColor;

//---------------------------------------------------------------------------
implementation

//---------------------------------------------------------------------------
function MinColor(const c1, c2, c3: TFixedColor): TFixedColor;
begin
 Result.r:= Min3(c1.r, c2.r, c3.r);
 Result.g:= Min3(c1.g, c2.g, c3.g);
 Result.b:= Min3(c1.b, c2.b, c3.b);
end;

//---------------------------------------------------------------------------
function MaxColor(const c1, c2, c3: TFixedColor): TFixedColor;
begin
 Result.r:= Max3(c1.r, c2.r, c3.r);
 Result.g:= Max3(c1.g, c2.g, c3.g);
 Result.b:= Max3(c1.b, c2.b, c3.b);
end;

//---------------------------------------------------------------------------
function ClampMinMax(Value, Min, Max: Integer): Integer;
asm { params: eax, edx, ecx }
 cmp eax, edx
 cmovl eax, edx
 cmp eax, ecx
 cmovg eax, ecx
end;

//---------------------------------------------------------------------------
function ClampColor(const Color, MinCol, MaxCol: TFixedColor): TFixedColor;
begin
 Result.r:= ClampMinMax(Color.r, MinCol.r, MaxCol.r);
 Result.g:= ClampMinMax(Color.g, MinCol.g, MaxCol.g);
 Result.b:= ClampMinMax(Color.b, MinCol.b, MaxCol.b);
{ Result:= Color;

 if (Result.r < MinCol.r) then Result.r:= MinCol.r;
 if (Result.g < MinCol.g) then Result.g:= MinCol.g;
 if (Result.b < MinCol.b) then Result.b:= MinCol.b;
 if (Result.r > MaxCol.r) then Result.r:= MaxCol.r;
 if (Result.g > MaxCol.g) then Result.g:= MaxCol.g;
 if (Result.b > MaxCol.b) then Result.b:= MaxCol.b;}
end;

//---------------------------------------------------------------------------
// Clamp()
//
// Clamps integer value between values of 0 and $FF00
//---------------------------------------------------------------------------
function Clamp8(Value: Integer): Integer;
{asm
 mov   edx, $3FC00
 cmp   edx, eax
 cmovl eax, edx
 xor   edx, edx
 cmp   edx, eax
 cmovg eax, edx}
begin
 Result:= Value;
 if (Result < 0) then Result:= 0;
 if (Result > $3FC00) then Result:= $3FC00;

end;

//---------------------------------------------------------------------------
class operator TFixedColor.Add(const a, b: TFixedColor): TFixedColor;
begin
 Result.r:= a.r + b.r;
 Result.g:= a.g + b.g;
 Result.b:= a.b + b.b;
end;

//---------------------------------------------------------------------------
class operator TFixedColor.Subtract(const a, b: TFixedColor): TFixedColor;
begin
 Result.r:= a.r - b.r;
 Result.g:= a.g - b.g;
 Result.b:= a.b - b.b;
end;

//---------------------------------------------------------------------------
procedure TFixedColor.Clamp();
begin
 r:= Clamp8(r);
 g:= Clamp8(g);
 b:= Clamp8(b);
end;

//---------------------------------------------------------------------------
class operator TFixedColor.Implicit(c: Cardinal): TFixedColor;
begin
 Result.r:= (c and $FF) * 256;
 Result.g:= ((c shr 8) and $FF) * 256;
 Result.b:= ((c shr 16) and $FF) * 256;
{ Result.r:= (c and $FF) shl 8;
 Result.g:= c and $FF00;
 Result.b:= (c shr 8) and $FF00;}
end;

//---------------------------------------------------------------------------
class operator TFixedColor.Implicit(const c: TFixedColor): Cardinal;
//var
// Aux: TFixedColor;
begin
// Aux:= c;
// Aux.Clamp;
// Result:= (Aux.r div 1024) or ((Aux.g div 1024) shl 8) or ((Aux.b div 1024) shl 16);
// Result:= (c.r div 1024) or ((c.g div 1024) shl 8) or ((c.b div 1024) shl 16);
 Result:= (c.r shr 8) or ((c.g shr 8) shl 8) or ((c.b shr 8) shl 16);
// (c.r shr 8) or (c.g and $FF00) or ((c.b and $FF00) shl 8);
end;

//---------------------------------------------------------------------------
end.
