unit FluxScene;

//---------------------------------------------------------------------------
interface

//---------------------------------------------------------------------------
uses
 Types, Math, Vectors2, Vectors4, Matrices4, FluxUtils, FluxMatrix,
 FluxMeshFaces, FluxMeshes4, SystemSurfaces, TexClipper2;

//---------------------------------------------------------------------------
const
 zNearPlane   = 1.0;

 MaxVertices  = 32768;
 MaxTriangles = 16384;
 DefaultClipRect: TRect = (Left: 0; Top: 0; Right: 160; Bottom: 100);

//---------------------------------------------------------------------------
type
 TDepthOrder = record
  Index: Integer;
  MidPt: PVector4;
 end;

//---------------------------------------------------------------------------
 PFluxTri = ^TFluxTri;
 TFluxTri = record
  vIndex : array[0..2] of Integer;
  uvIndex: array[0..2] of Integer;
  Visible: Boolean;
 end;

//---------------------------------------------------------------------------
 PVertexArray = ^TVertexArray;
 TVertexArray = array[0..MaxVertices - 1] of TVector4;

//---------------------------------------------------------------------------
 PTriArray = ^TTriArray;
 TTriArray = array[0..MaxTriangles - 1] of TVector4;

//---------------------------------------------------------------------------
 TFluxScene = class
 private
  PreVertices  : PVertexArray;
  PostVertices : PVertexArray;
  Normals      : PVertexArray;
  TextureCoords: array[0..MaxVertices - 1] of TPoint2;
  VertexCount  : Integer;
  TextureCount : Integer;

  MidPoints    : PTriArray;
  DepthOrder   : array[0..MaxTriangles - 1] of TDepthOrder;
  Triangles    : array[0..MaxTriangles - 1] of TFluxTri;
  TriangleCount: Integer;

  AuxMatrix    : TFluxMatrix;
  FFieldOfView : Single;
  FAspectRatio : Single;
  
  FShiftTex: TPoint2;
  FAffineMapping: Boolean;

  procedure BackfaceCull();
  procedure ZPlaneCull();
  procedure QuickSort(Left, Right: Integer);
  procedure DepthSort();
  procedure Project(const Size: TPoint2);
  procedure RenderTriangles(Surface, Texture: TSystemSurface);
 public
  property FieldOfView: Single read FFieldOfView write FFieldOfView;
  property AspectRatio: Single read FAspectRatio write FAspectRatio;

  property AffineMapping: Boolean read FAffineMapping write FAffineMapping;

  property TotalVertices : Integer read VertexCount;
  property TotalTriangles: Integer read TriangleCount;

  property ShiftTex: TPoint2 read FShiftTex write FShiftTex;

  procedure BeginScene();
  procedure Draw(Mesh: TFluxMesh4; WorldMtx: PMatrix4);
  procedure EndScene(ViewMtx: PMatrix4);
  procedure Present(Surface, Texture: TSystemSurface);

  constructor Create();
  destructor Destroy(); override;
 end;

//---------------------------------------------------------------------------
implementation

//---------------------------------------------------------------------------
procedure TransferV4(Source: TVectors4; Dest, Matrix: Pointer);
begin
 BatchTransform1(Source.MemAddr, Dest, (Source.Count + 1) div 2, Matrix);
// BatchMultiply5(Source.MemAddr, Dest, (Source.Count + 1) div 2, Matrix);
end;

//---------------------------------------------------------------------------
procedure TransferF4(Source: PFluxMeshFace; Dest: PFluxTri; Count,
 vAdd, uvAdd: Integer);
var
 i: Integer;
begin
 for i:= 0 to Count - 1 do
  begin
   Dest.vIndex[0]:= Source.vIndex[0] + vAdd;
   Dest.vIndex[1]:= Source.vIndex[1] + vAdd;
   Dest.vIndex[2]:= Source.vIndex[2] + vAdd;

   Dest.uvIndex[0]:= Source.uvIndex[0] + uvAdd;
   Dest.uvIndex[1]:= Source.uvIndex[1] + uvAdd;
   Dest.uvIndex[2]:= Source.uvIndex[2] + uvAdd;

   Inc(Source);
   Inc(Dest);
  end;
end;

//---------------------------------------------------------------------------
constructor TFluxScene.Create();
begin
 inherited;

 SetMinimumBlockAlignment(mba16Byte);
 PreVertices := AllocMem(SizeOf(TVertexArray));
 PostVertices:= AllocMem(SizeOf(TVertexArray));
 Normals     := AllocMem(SizeOf(TVertexArray));
 MidPoints   := AllocMem(SizeOf(TTriArray));
 SetMinimumBlockAlignment(mba8Byte);

 AuxMatrix:= TFluxMatrix.Create();

 FFieldOfView:= 45.0;
 FAspectRatio:= 1.0;
end;

//---------------------------------------------------------------------------
destructor TFluxScene.Destroy();
begin
 AuxMatrix.Free();

 FreeMem(MidPoints);
 FreeMem(Normals);
 FreeMem(PostVertices);
 FreeMem(PreVertices);

 inherited;
end;

//---------------------------------------------------------------------------
procedure TFluxScene.BeginScene();
begin
 VertexCount  := 0;
 TriangleCount:= 0;
 TextureCount := 0;
end;

//---------------------------------------------------------------------------
procedure TFluxScene.Draw(Mesh: TFluxMesh4; WorldMtx: PMatrix4);
var
 i: Integer; 
begin
 // transfer vertices
 TransferV4(Mesh.Vertices, @PreVertices[VertexCount], WorldMtx);

 // transfer mid-points
 TransferV4(Mesh.FaceMidPts, @MidPoints[TriangleCount], WorldMtx);

 // apply rotation only
 AuxMatrix.LoadRotation(WorldMtx);

 // transfer normals
 TransferV4(Mesh.Normals, @Normals[VertexCount], AuxMatrix.RawMtx);

 // transfer faces
 TransferF4(Mesh.Faces.Face[0], @Triangles[TriangleCount],
  Mesh.Faces.Count, VertexCount, TextureCount);

 // transfer texture coordinates
 for i:= 0 to Mesh.TexCoords.Count - 1 do
  TextureCoords[TextureCount + i]:= Mesh.TexCoords[i]^ + FShiftTex;

 // update pointers
 Inc(VertexCount, Mesh.Vertices.Count);
 Inc(TriangleCount, Mesh.Faces.Count);
 Inc(TextureCount, Mesh.TexCoords.Count);
end;

//---------------------------------------------------------------------------
procedure TFluxScene.BackfaceCull();
var
 i: Integer;
 Tri: PFluxTri;
 p2: PVector4;
 Normal: TVector4;
begin
 for i:= 0 to TriangleCount - 1 do
  begin
   Tri:= @Triangles[i];
   p2 := @PostVertices[Tri.vIndex[2]];

   Normal:= Cross4(p2^ - PostVertices[Tri.vIndex[0]], p2^ -
    PostVertices[Tri.vIndex[1]]);

  Triangles[i].Visible:= (Dot4(p2^, Normal) <= 0.0);
 end;
end;

//---------------------------------------------------------------------------
procedure TFluxScene.ZPlaneCull();
var
 i, j: Integer;
 Tri: PFluxTri;
 Vertex: PVector4;
 zMin: Single;
begin
 Tri:= @Triangles[0];

 for i:= 0 to TriangleCount - 1 do
  begin
   if (Tri.Visible) then
    begin
     zMin:= PostVertices[Tri.vIndex[2]].z;

     for j:= 0 to 1 do
      begin
       Vertex:= @PostVertices[Tri.vIndex[j]];
       if (Vertex.z < zMin) then zMin:= Vertex.z;
      end;

     Tri.Visible:= (zMin > zNearPlane);
    end;

   Inc(Tri);
  end;
end;

//---------------------------------------------------------------------------
procedure TFluxScene.EndScene(ViewMtx: PMatrix4);
begin
 if (VertexCount < 1)or(TriangleCount < 1) then Exit;

 // transform vertices
 BatchTransform1(@PreVertices[0], @PostVertices[0], (VertexCount + 1) div 2,
  ViewMtx);

 // trasnform mid-points
 BatchTransform1(@MidPoints[0], @MidPoints[0], (TriangleCount + 1) div 2,
  ViewMtx);

 // apply rotation only
 AuxMatrix.LoadRotation(ViewMtx);

 // apply backface culling
 BackfaceCull();

 // apply z-plane culling
 ZPlaneCull();
end;

//---------------------------------------------------------------------------
procedure TFluxScene.QuickSort(Left, Right: Integer);
var
 i, j: Integer;
 Aux: TDepthOrder;
 z: Single;
begin
 i:= Left;
 j:= Right;
 z:= DepthOrder[(Left + Right) shr 1].MidPt.z;

 repeat
  while (DepthOrder[i].MidPt.z < z) do Inc(i);
  while (z < DepthOrder[j].MidPt.z) do Dec(j);

  if (i <= j) then
   begin
    Aux:= DepthOrder[i];
    DepthOrder[i]:= DepthOrder[j];
    DepthOrder[j]:= Aux;

    Inc(i);
    Dec(j);
   end;
 until (i > j);

 if (Left < j) then QuickSort(Left, j);
 if (i < Right) then QuickSort(i, Right);
end;

//---------------------------------------------------------------------------
procedure TFluxScene.DepthSort();
var
 i: Integer;
begin
 if (TriangleCount < 1) then Exit;

 for i:= 0 to TriangleCount - 1 do
  begin
   DepthOrder[i].Index:= i;
   DepthOrder[i].MidPt:= @MidPoints[i];
  end;

 QuickSort(0, TriangleCount - 1);
end;

//---------------------------------------------------------------------------
procedure TFluxScene.Project(const Size: TPoint2);
var
 Delta, DeltaX, DeltaY: Single;
 i: Integer;
 Vertex: PVector4;
begin
 // Calculate distance to projection plane
 Delta := Cot(FFieldOfView * 0.5);
 DeltaX:= (Delta * FAspectRatio) * Size.X;
 DeltaY:= -Delta * Size.Y;

 // Project vertices
 Vertex:= @PostVertices[0];
 for i:= 0 to VertexCount - 1 do
  begin
   if (Vertex.z >= ZNearPlane) then
    begin
     Vertex.x:= ((DeltaX * Vertex.x) / Vertex.z) + Size.x;
     Vertex.y:= ((DeltaY * Vertex.y) / Vertex.z) + Size.y;
    end;

   Inc(Vertex);
  end;
end;

//---------------------------------------------------------------------------
procedure TFluxScene.RenderTriangles(Surface, Texture: TSystemSurface);
var
 i, j: Integer;
 Vertices: array[0..2] of PVector4;
 TexCoord: array[0..2] of PPoint2;
 Tri : PFluxTri;
begin
 for j:= TriangleCount - 1 downto 0 do
  begin
   Tri:= @Triangles[DepthOrder[j].Index];
   if (not Tri.Visible) then Continue;

   for i:= 0 to 2 do
    begin
     Vertices[i]:= @PostVertices[Tri.vIndex[i]];
     TexCoord[i]:= @TextureCoords[Tri.uvIndex[i]];
    end;

   if (FAffineMapping) then
    begin
     TexMapAffine(Surface, Texture, Vertices[0]^, Vertices[1]^, Vertices[2]^,
      TexCoord[0]^, TexCoord[1]^, TexCoord[2]^);
    end else
    begin
     TexMapPerspective(Surface, Texture, Vertices[0]^, Vertices[1]^,
      Vertices[2]^, TexCoord[0]^, TexCoord[1]^, TexCoord[2]^);
    end;
  end;
end;

//---------------------------------------------------------------------------
procedure TFluxScene.Present(Surface, Texture: TSystemSurface);
begin
 DepthSort();

 Project(Point2(Surface.Width * 0.5, Surface.Height * 0.5));

 RenderTriangles(Surface, Texture);
end;

//---------------------------------------------------------------------------
end.
