unit MainFm;

//---------------------------------------------------------------------------
interface

//---------------------------------------------------------------------------
uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, SystemSurfaces, Vectors2, FluxScene, FluxMeshes4, FluxMatrix;

//---------------------------------------------------------------------------
type
  TMainForm = class(TForm)
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
  private
    { Private declarations }
    Texture: TSystemSurface;
    Display: TSystemSurface;
    Bitmap : TBitmap;

    Ticks: Integer;

    Scene: TFluxScene;
    Mesh : TFluxMesh4;

    ViewMtx : TFluxMatrix;
    WorldMtx: TFluxMatrix;

    procedure TimerEvent(Sender: TObject);
    procedure ProcessEvent(Sender: TObject);
  public
    { Public declarations }
  end;

//---------------------------------------------------------------------------
var
  MainForm: TMainForm;

//---------------------------------------------------------------------------
implementation
uses
 Vectors3, TexClipper2, AsphyreTimer, FluxTorusKnot;
{$R *.dfm}

//---------------------------------------------------------------------------
procedure TMainForm.FormCreate(Sender: TObject);
begin
 Display:= TSystemSurface.Create();
 Display.SetSize(640, 480);

 Texture:= TSystemSurface.Create();
 Texture.LoadFromTGA('texture.tga');

 Bitmap:= TBitmap.Create();
 Bitmap.PixelFormat:= pf32bit;
 Bitmap.SetSize(640, 480);

 Scene:= TFluxScene.Create();
 Scene.AspectRatio:= 480.0 / 640.0;

 Mesh:= MakeTorusPQ(50.0, 10.0, 3, 2, 128, 16, 8.0, 1.0);

 WorldMtx:= TFluxMatrix.Create();
 ViewMtx := TFluxMatrix.Create();

 Timer.OnTimer  := TimerEvent;
 Timer.OnProcess:= ProcessEvent;
 Timer.Enabled:= True;
 Timer.MaxFPS:= 4000;
end;

//---------------------------------------------------------------------------
procedure TMainForm.FormDestroy(Sender: TObject);
begin
 FreeAndNil(ViewMtx);
 FreeAndNil(WorldMtx);
 FreeAndNil(Mesh);
 FreeAndNil(Scene);
 FreeAndNil(Bitmap);
 FreeAndNil(Texture);
 FreeAndNil(Display);
end;

//---------------------------------------------------------------------------
procedure TMainForm.FormPaint(Sender: TObject);
var
 i: Integer;
begin
 ViewMtx.LoadIdentity();
 ViewMtx.LookAt(Vector3(0.0, 10.0 - Sin(Ticks * 0.0071) * 30.0,
  -200.0 + Cos(Ticks * 0.01) * 50.0), ZeroVec3, AxisYVec3);

 WorldMtx.LoadIdentity();
 WorldMtx.RotateY(Ticks * 0.01);

 Display.Clear($FF343126);

 Scene.ShiftTex:= Point2(-Ticks * 0.005, Ticks * 0.002);

 Scene.BeginScene();
 Scene.Draw(Mesh, WorldMtx.RawMtx);
 Scene.EndScene(ViewMtx.RawMtx);
 Scene.Present(Display, Texture);

 for i:= 0 to Display.Height - 1 do
  Move(Display.Scanline[i]^, Bitmap.Scanline[i]^, Display.Width * 4);

 with Bitmap.Canvas do
  begin
   Font.Name:= 'Tahoma';
   Font.Size:= 9;
   Font.Style:= [fsBold];
   Font.Color:= $9EC7D3;

   Brush.Style:= bsClear;

   TextOut(4, 4, 'FPS: ' + IntToStr(Timer.FrameRate));

   Font.Color:= $FFFFFF;

   if (Scene.AffineMapping) then
    TextOut(4, 24, 'Using Affine Texture Mapping.')
     else TextOut(4, 24, 'Using Perspective Texture Mapping.');

   Font.Color:= $E0E0E0;

   TextOut(4, 44, '(press Space to change)');

   Font.Color:= $73FEC8;

   TextOut(4, 64, 'Drawing ' + IntToStr(Scene.TotalVertices) +
    ' vertices and ' + IntToStr(Scene.TotalTriangles) + ' triangles.');
  end;

 Canvas.Draw(0, 0, Bitmap);
end;

//---------------------------------------------------------------------------
procedure TMainForm.TimerEvent(Sender: TObject);
begin
 FormPaint(nil);

 Timer.Process();
end;

//---------------------------------------------------------------------------
procedure TMainForm.ProcessEvent(Sender: TObject);
begin
 Inc(Ticks);
end;

//---------------------------------------------------------------------------
procedure TMainForm.FormKeyDown(Sender: TObject; var Key: Word;
 Shift: TShiftState);
begin
 if (Key = VK_SPACE) then
  Scene.AffineMapping:= not Scene.AffineMapping;
end;

//---------------------------------------------------------------------------
end.
