unit FluxFaces;
//---------------------------------------------------------------------------
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is FluxFaces.pas.
//
// The Initial Developer of the Original Code is M. Sc. Yuriy Kotsarenko.
// Portions created by M. Sc. Yuriy Kotsarenko are Copyright (C) 2007,
// M. Sc. Yuriy Kotsarenko. All Rights Reserved.
//---------------------------------------------------------------------------
interface

//---------------------------------------------------------------------------
uses
 Classes;

//---------------------------------------------------------------------------
const
 ffaTextured   = $01;
 ffaSelfLight  = $02;
 ffaUnoccluded = $04;
 ffaCulled     = $08;

//---------------------------------------------------------------------------
type
 PFluxFace = ^TFluxFace;
 TFluxFace = record
  vIndex  : array[0..2] of Integer;
  uvIndex : array[0..2] of Integer;
  Material: Integer;
  Flags   : Cardinal;
 end;

//---------------------------------------------------------------------------
 TFluxFaces = class
 private
  Data: array of TFluxFace;
  DataCount: Integer;

  procedure Request(Amount: Integer);
  function GetItem(Num: Integer): TFluxFace;
  procedure SetItem(Num: Integer; const Value: TFluxFace);
  function GetTriangle(Num: Integer): PFluxFace;
 public
  property Count: Integer read DataCount;
  property Items[Num: Integer]: TFluxFace read GetItem write SetItem; default;
  property Triangle[Num: Integer]: PFluxFace read GetTriangle;

  function Insert(const Tri: TFluxFace): Integer; overload;
  function Insert(v0, v1, v2, uv0, uv1, uv2: Integer): Integer; overload;
  function Insert(v0, v1, v2: Integer): Integer; overload;
  procedure Remove(Num: Integer);
  procedure RemoveAll();

  procedure SetFlags(Flags: Cardinal);

  procedure AddFrom(Source: TFluxFaces; vIndexAdd, uvIndexAdd: Integer);
  procedure LoadFromStream(Stream: TStream);
  procedure SaveToStream(Stream: TStream);
 end;

//---------------------------------------------------------------------------
implementation

//---------------------------------------------------------------------------
const
 CacheSize = 256;

//--------------------------------------------------------------------------
procedure TFluxFaces.Request(Amount: Integer);
var
 Required: Integer;
begin
 Required:= ((Amount + CacheSize - 1) div CacheSize) * CacheSize;
 if (Required > Length(Data)) then SetLength(Data, Required);
end;

//---------------------------------------------------------------------------
function TFluxFaces.GetItem(Num: Integer): TFluxFace;
begin
 Assert((Num >= 0)and(Num < DataCount),
  'TFluxFaces.GetItem: Index out of bounds');

 Result:= Data[Num];
end;

//---------------------------------------------------------------------------
procedure TFluxFaces.SetItem(Num: Integer; const Value: TFluxFace);
begin
 Assert((Num >= 0)and(Num < DataCount),
  'TFluxFaces.SetItem: Index out of bounds');

 Data[Num]:= Value;
end;

//---------------------------------------------------------------------------
function TFluxFaces.GetTriangle(Num: Integer): PFluxFace;
begin
 Assert((Num >= 0)and(Num < DataCount),
  'TFluxFaces.GetTriangle: Index out of bounds');

 Result:= @Data[Num];
end;

//---------------------------------------------------------------------------
function TFluxFaces.Insert(const Tri: TFluxFace): Integer;
var
 Index: Integer;
begin
 Request(DataCount + 1);

 Index:= DataCount;
 Data[Index]:= Tri;
 Inc(DataCount);

 Result:= Index;
end;

//---------------------------------------------------------------------------
function TFluxFaces.Insert(v0, v1, v2, uv0, uv1, uv2: Integer): Integer;
var
 Tri: TFluxFace;
begin
 Tri.vIndex[0] := v0;
 Tri.vIndex[1] := v1;
 Tri.vIndex[2] := v2;
 Tri.uvIndex[0]:= uv0;
 Tri.uvIndex[1]:= uv1;
 Tri.uvIndex[2]:= uv2;

 Result:= Insert(Tri);
end;

//---------------------------------------------------------------------------
function TFluxFaces.Insert(v0, v1, v2: Integer): Integer;
begin
 Result:= Insert(v0, v1, v2, v0, v1, v2);
end;

//---------------------------------------------------------------------------
procedure TFluxFaces.Remove(Num: Integer);
var
 i: Integer;
begin
 Assert((Num >= 0)and(Num < DataCount),
  'TFluxFaces.Remove: Index out of bounds');

 for i:= Num to DataCount - 2 do
  Data[i]:= Data[i + 1];

 Dec(DataCount);
end;

//---------------------------------------------------------------------------
procedure TFluxFaces.RemoveAll();
begin
 DataCount:= 0;
end;

//---------------------------------------------------------------------------
procedure TFluxFaces.SetFlags(Flags: Cardinal);
var
 i: Integer;
begin
 for i:= 0 to DataCount - 1 do
  Data[i].Flags:= Flags;
end;

//---------------------------------------------------------------------------
procedure TFluxFaces.AddFrom(Source: TFluxFaces; vIndexAdd,
 uvIndexAdd: Integer);
var
 i, SrcIndex, NewAmount: Integer;
begin
 NewAmount:= DataCount + Source.DataCount;
 Request(NewAmount);

 SrcIndex:= 0;
 for i:= DataCount to NewAmount - 1 do
  begin
   Data[i]:= Source.Data[SrcIndex];
   with Data[i] do
    begin
     vIndex[0] := vIndex[0]  + vIndexAdd;
     vIndex[1] := vIndex[1]  + vIndexAdd;
     vIndex[2] := vIndex[2]  + vIndexAdd;
     uvIndex[0]:= uvIndex[0] + uvIndexAdd;
     uvIndex[1]:= uvIndex[1] + uvIndexAdd;
     uvIndex[2]:= uvIndex[2] + uvIndexAdd;
    end;

   Inc(SrcIndex);
  end;

 DataCount:= NewAmount;
end;

//--------------------------------------------------------------------------
procedure TFluxFaces.SaveToStream(Stream: TStream);
var
 i: Integer;
begin
 Stream.WriteBuffer(DataCount, SizeOf(Integer));

 // only save vIndex, uvIndex and TPoint3 part of Normal ("w" discarded)
 for i:= 0 to DataCount - 1 do
  Stream.WriteBuffer(Data[i], 6 * SizeOf(Integer));
end;

//--------------------------------------------------------------------------
procedure TFluxFaces.LoadFromStream(Stream: TStream);
var
 i: Integer;
begin
 Stream.ReadBuffer(DataCount, SizeOf(Integer));
 Request(DataCount);

 // only read vIndex, uvIndex and TPoint3 part of Normal
 for i:= 0 to DataCount - 1 do
  begin
   FillChar(Data[i], SizeOf(TFluxFace), 0);
   Stream.ReadBuffer(Data[i], 6 * SizeOf(Integer));
  end;
end;

//---------------------------------------------------------------------------
end.
