unit FluxMeshes;
//---------------------------------------------------------------------------
// The contents of this file are subject to the Mozilla Public License
// Version 1.1 (the "License"); you may not use this file except in
// compliance with the License. You may obtain a copy of the License at
// http://www.mozilla.org/MPL/
//
// Software distributed under the License is distributed on an "AS IS"
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
// License for the specific language governing rights and limitations
// under the License.
//
// The Original Code is FluxMeshes.pas.
//
// The Initial Developer of the Original Code is M. Sc. Yuriy Kotsarenko.
// Portions created by M. Sc. Yuriy Kotsarenko are Copyright (C) 2007,
// M. Sc. Yuriy Kotsarenko. All Rights Reserved.
//---------------------------------------------------------------------------
interface

//---------------------------------------------------------------------------
uses
 Classes, SysUtils, AsphyreDb, Vectors2, Vectors3, FluxMeshFaces;

//---------------------------------------------------------------------------
type
 TFluxMesh = class
 private
  FName       : string;
  FVertices   : TVectors3;
  FNormals    : TVectors3;
  FTexCoords  : TPoints2;
  FFaceNormals: TVectors3;
  FFaceMidPts : TVectors3;
  FFaces      : TFluxMeshFaces;
 public
  // The name of the mesh
  property Name: string read FName write FName;

  // primary components
  property Vertices : TVectors3 read FVertices;
  property Normals  : TVectors3 read FNormals;
  property TexCoords: TPoints2 read FTexCoords;

  // secondary components
  property FaceNormals: TVectors3 read FFaceNormals;
  property FaceMidPts : TVectors3 read FFaceMidPts;

  // mesh faces
  property Faces: TFluxMeshFaces read FFaces;

  // manually calculates face mid-points
  procedure FindFaceMidPoints();

  // manually find face normals
  procedure FindFaceNormals();

  // manually find both face and vertex normals
  procedure FindAllNormals();

  function SaveToStream(Stream: TStream): Boolean;
  function LoadFromStream(Stream: TStream): Boolean;

  function SaveToFile(Filename: string): Boolean;
  function LoadFromFile(Filename: string): Boolean;
  function LoadFromASDb(const Key: string; ASDb: TASDb): Boolean;

  constructor Create();
  destructor Destroy(); override;
 end;

//---------------------------------------------------------------------------
implementation

//---------------------------------------------------------------------------
constructor TFluxMesh.Create();
begin
 inherited Create();

 FVertices := TVectors3.Create();
 FNormals  := TVectors3.Create();
 FTexCoords:= TPoints2.Create();

 FFaceNormals:= TVectors3.Create();
 FFaceMidPts := TVectors3.Create();

 FFaces:= TFluxMeshFaces.Create();
end;

//---------------------------------------------------------------------------
destructor TFluxMesh.Destroy();
begin
 FFaces.Free();
 FFaceMidPts.Free();
 FFaceNormals.Free();

 FTexCoords.Free();
 FNormals.Free();
 FVertices.Free();

 inherited;
end;

//---------------------------------------------------------------------------
procedure TFluxMesh.FindFaceMidPoints();
var
 i: Integer;
 Face: PFluxMeshFace;
begin
 FFaceMidPts.RemoveAll();

 for i:= 0 to FFaces.Count - 1 do
  begin
   Face:= FFaces.Face[i];

   FFaceMidPts.Add((FVertices[Face.vIndex[0]] + FVertices[Face.vIndex[1]] +
    FVertices[Face.vIndex[2]]) / 3.0);
  end;
end;

//---------------------------------------------------------------------------
procedure TFluxMesh.FindFaceNormals();
var
 i: Integer;
 Face: PFluxMeshFace;
 v0, v1: TVector3;
begin
 FFaceNormals.RemoveAll();

 for i:= 0 to FFaces.Count - 1 do
  begin
   Face:= Faces.Face[i];

   v0:= Vertices[Face.vIndex[2]] - Vertices[Face.vIndex[0]];
   v1:= Vertices[Face.vIndex[2]] - Vertices[Face.vIndex[1]];
   FFaceNormals.Add(Norm3(Cross3(v0, v1)))
  end;
end;

//--------------------------------------------------------------------------
procedure TFluxMesh.FindAllNormals();
const
 WeldEpsilon = 0.00001;
var
 Weights: array of Real;
 i, FaceNo, VertNo: Integer;
 Face: PFluxMeshFace;
 v0, v1, Normal: TVector3;
begin
 // remove existing normal information
 FNormals.RemoveAll();
 FFaceNormals.RemoveAll();

 // define the size of weight array
 SetLength(Weights, Faces.Count);

 // find face normals and weights
 for FaceNo:= 0 to Faces.Count - 1 do
  begin
   Face:= Faces.Face[FaceNo];

   v0:= Vertices[Face.vIndex[2]] - Vertices[Face.vIndex[0]];
   v1:= Vertices[Face.vIndex[2]] - Vertices[Face.vIndex[1]];

   Normal:= Cross3(v0, v1);

   FFaceNormals.Add(Norm3(Normal));
   Weights[FaceNo]:= Length3(Normal);
  end;

 // find vertex normals
 for i:= 0 to FVertices.Count - 1 do
  begin
   Normal:= ZeroVec3;
   v0    := FVertices[i];

   for FaceNo:= 0 to Faces.Count - 1 do
    for VertNo:= 0 to 2 do
     begin
      v1:= FVertices[Faces[FaceNo].vIndex[VertNo]];

      if (Length3(v1 - v0) < WeldEpsilon) then
       Normal:= Normal + (FFaceNormals[FaceNo] * Weights[FaceNo]);
     end;

   FNormals.Add(Norm3(Normal));
  end;
end;

//---------------------------------------------------------------------------
function TFluxMesh.SaveToStream(Stream: TStream): Boolean;
begin
 Result:= True;

 try
  FVertices.SaveToStream(Stream);
  FNormals.SaveToStream(Stream);
  FTexCoords.SaveToStream(Stream);
  FFaceNormals.SaveToStream(Stream);
  FFaceMidPts.SaveToStream(Stream);
  FFaces.SaveToStream(Stream);
 except
  Result:= False;
 end;
end;

//---------------------------------------------------------------------------
function TFluxMesh.LoadFromStream(Stream: TStream): Boolean;
begin
 Result:= True;

 try
  FVertices.LoadFromStream(Stream);
  FNormals.LoadFromStream(Stream);
  FTexCoords.LoadFromStream(Stream);
  FFaceNormals.LoadFromStream(Stream);
  FFaceMidPts.LoadFromStream(Stream);
  FFaces.LoadFromStream(Stream);
 except
  Result:= False;
 end;
end;

//--------------------------------------------------------------------------
function TFluxMesh.SaveToFile(Filename: string): Boolean;
var
 Stream: TFileStream;
begin
 Stream:= TFileStream.Create(Filename, fmCreate or fmShareExclusive);

 try
  Result:= SaveToStream(Stream);
 finally
  Stream.Free();
 end;
end;

//--------------------------------------------------------------------------
function TFluxMesh.LoadFromFile(Filename: string): Boolean;
var
 Stream: TFileStream;
begin
 Stream:= TFileStream.Create(Filename, fmOpenRead or fmShareDenyWrite);

 try
  Result:= LoadFromStream(Stream);
 finally
  Stream.Free();
 end;
end;

//---------------------------------------------------------------------------
function TFluxMesh.LoadFromASDb(const Key: string; ASDb: TASDb): Boolean;
var
 Stream: TMemoryStream;
begin
 Stream:= TMemoryStream.Create();
 Result:= ASDb.ReadStream(Key, Stream);
 if (not Result) then
  begin
   Stream.Free();
   Exit;
  end;

 try
  Stream.Seek(0, soFromBeginning);
  Result:= LoadFromStream(Stream);
 except
  Result:= False;
 end;

 Stream.Free();  
end;

//---------------------------------------------------------------------------
end.
