unit TxtArb2;
//---------------------------------------------------------------------------
// TxtArb2.pas                                          Modified: 03-Dec-2007
// Arbitrary Size Texture Mapper
//
// This code is based on C/C++ affine texture mapping code originally
// published in "fatmap2.zip" package by Mats Byggmastar, 1997.
//
// The original articles and source code can be downloaded at:
//  http://chrishecker.com/Miscellaneous_Technical_Articles
//
// The conversion and adaptation was made by Yuriy Kotsarenko, Nov 2007.
//---------------------------------------------------------------------------
interface

//---------------------------------------------------------------------------
uses
 Vectors2, Vectors3, SystemSurfaces, AsphyreUtils;

//---------------------------------------------------------------------------
const
 // The following values are used for texture coordinate wrapping, when
 // (u, v) values are outside [0..1] range.
 //
 // These values should be set to (Width - 1, Height - 1).
 WrapUAnd = $1FF;
 WrapVAnd = $1FF;

//---------------------------------------------------------------------------
// Notice: the vertices should be specified in anti-clockwise order.
//---------------------------------------------------------------------------
procedure TexMap(Dest, Texture: TSystemSurface; const v0, v1, v2: TVector3;
 uv0, uv1, uv2: TPoint2);

//---------------------------------------------------------------------------
implementation

//---------------------------------------------------------------------------
type
 PVertexUV = ^TVertexUV;
 TVertexUV = record
  x, y: Integer; // Screen position in 16:16 bit fixed point
  u, v: Integer; // Texture u,v in 16:16 bit fixed point
 end;

//---------------------------------------------------------------------------
var
 max_vtx: PVertexUV;             // Max y vertex (ending vertex)
 start_vtx, end_vtx: PVertexUV;  // First and last vertex in array
 right_vtx, left_vtx: PVertexUV; // Current right and left vertex

 right_height, left_height: Integer;
 right_x, right_dxdy, left_x, left_dxdy: Integer;
 left_u, left_dudy, left_v, left_dvdy: Integer;
 dudxfrac, dvdxfrac: Integer;
 duvdxstep: array[0..1] of Integer;

 DestBits : Pointer;
 DestPitch: Integer;

//---------------------------------------------------------------------------
function Ceil(x: Integer): Integer; inline;
begin
 Result:= (x + $FFFF) shr 16;
end;

//---------------------------------------------------------------------------
function iMul14(x, y: Integer): Integer;
{$ifdef fpc} assembler;{$endif}
asm { params: eax, edx }
 imul edx
 shrd eax, edx, 14
end;

//---------------------------------------------------------------------------
function FloatToFixed(Value: Single): Integer;
begin
 Result:= Round(Value * 65536.0);
end;

//---------------------------------------------------------------------------
procedure Inner(Dest, TextureBits: Pointer; Width: Integer;
 u, v, du, dv: Integer; TexturePitch: Integer);
var
 i: Integer;
begin
 for i:= 0 to Width - 1 do
  begin
   PCardinal(Dest)^:= PCardinal(Integer(TextureBits) +
    ((u shr 16) and WrapUAnd) * 4 +
    ((v shr 16) and WrapVAnd) * TexturePitch)^;
   Inc(Integer(Dest), 4);

   Inc(u, du);
   Inc(v, dv);
  end;
end;

//---------------------------------------------------------------------------
procedure RightSection();
var
 v1, v2: PVertexUV;
 Height, Inv_Height, Prestep: Integer;
begin
 // Walk backwards trough the vertex array
 v1:= right_vtx;

 if (Integer(right_vtx) > Integer(start_vtx)) then
  v2:= Pointer(Integer(right_vtx) - SizeOf(TVertexUV)) else
   v2:= end_vtx; // Wrap to end of array

 right_vtx:= v2;

 // v1 = top vertex
 // v2 = bottom vertex

 // Calculate number of scanlines in this section

 right_height:= Ceil(v2.y) - Ceil(v1.y);
 if (right_height <= 0) then Exit;

 // Guard against possible div overflows

 if (right_height > 1) then
  begin
   // OK, no worries, we have a section that is at least
   // one pixel high. Calculate slope as usual.
   Height:= v2.y - v1.y;
   right_dxdy:= iDiv16(v2.x - v1.x, Height);
  end else
  begin
   // Height is less or equal to one pixel.
   // Calculate slope = width * 1/height
   // using 18:14 bit precision to avoid overflows.
   Inv_Height:= ($10000 shl 14) div (v2.y - v1.y);
   right_dxdy:= iMul14(v2.x - v1.x, Inv_Height);
  end;

 // Prestep initial values
 Prestep:= (Ceil(v1.y) shl 16) - v1.y;
 right_x:= v1.x + iMul16(Prestep, right_dxdy);
end;

//---------------------------------------------------------------------------
procedure LeftSection();
var
 v1, v2: PVertexUV;
 Height, Inv_Height, Prestep: Integer;
begin
 // Walk forward trough the vertex array
 v1:= left_vtx;

 if (Integer(left_vtx) < Integer(end_vtx)) then
  v2:= Pointer(Integer(left_vtx) + SizeOf(TVertexUV)) else
   v2:= start_vtx; // Wrap to start of array

 left_vtx:= v2;

 // v1 = top vertex
 // v2 = bottom vertex

 // Calculate number of scanlines in this section

 left_height:= Ceil(v2.y) - Ceil(v1.y);
 if (left_height <= 0) then Exit;

 // Guard against possible div overflows

 if (left_height > 1) then
  begin
   // OK, no worries, we have a section that is at least
   // one pixel high. Calculate slope as usual.
   Height:= v2.y - v1.y;
   left_dxdy:= iDiv16(v2.x - v1.x, Height);
   left_dudy:= iDiv16(v2.u - v1.u, Height);
   left_dvdy:= iDiv16(v2.v - v1.v, Height);
  end else
  begin
   // Height is less or equal to one pixel.
   // Calculate slope = width * 1/height
   // using 18:14 bit precision to avoid overflows.
   Inv_Height:= ($10000 shl 14) div (v2.y - v1.y);
   left_dxdy := iMul14(v2.x - v1.x, Inv_Height);
   left_dudy := iMul14(v2.u - v1.u, Inv_Height);
   left_dvdy := iMul14(v2.v - v1.v, Inv_Height);
  end;

 // Prestep initial values
 Prestep:= (Ceil(v1.y) shl 16) - v1.y;
 left_x := v1.x + iMul16(prestep, left_dxdy);
 left_u := v1.u + iMul16(prestep, left_dudy);
 left_v := v1.v + iMul16(prestep, left_dvdy);
end;

//---------------------------------------------------------------------------
procedure DrawArbitraryTexturePoly(vtx: PVertexUV; Vertices: Integer;
 TextureBits: Pointer; TexturePitch, dudx, dvdx: Integer);
var
 min_vtx: PVertexUV;
 min_y, max_y, n, x1, Width, Prestep, u, v: Integer;
 DestPtr: Pointer;
begin
 start_vtx:= vtx; // First vertex in array

 // Search trough the vtx array to find min y, max y
 // and the location of these structures.
 min_vtx:= vtx;
 max_vtx:= vtx;

 min_y:= vtx.y;
 max_y:= vtx.y;

 Inc(vtx);

 for n:= 1 to Vertices - 1 do
  begin
   if (vtx.y < min_y) then
    begin
     min_y  := vtx.y;
     min_vtx:= vtx;
    end else
   if(vtx.y > max_y) then
    begin
     max_y  := vtx.y;
     max_vtx:= vtx;
    end;
   Inc(vtx);
  end;

 // OK, now we know where in the array we should start and
 // where to end while scanning the edges of the polygon

 left_vtx := min_vtx; // Left side starting vertex
 right_vtx:= min_vtx; // Right side starting vertex
 end_vtx  := Pointer(Integer(vtx) - SizeOf(TVertexUV)); // Last vertex in array

 // Search for the first usable right section
 repeat
  if (right_vtx = max_vtx) then Exit;
  RightSection();
 until (not (right_height <= 0));

 // Search for the first usable left section
 repeat
  if (left_vtx = max_vtx) then Exit;
  LeftSection();
 until (not (left_height <= 0));

 DestPtr:= Pointer(Integer(DestBits) + Ceil(min_y) * DestPitch);

 dudxfrac:= dudx shl 16;      // Fractional part in high 16 bits
 dvdxfrac:= dvdx shl 16;      // clears low 16 bits also
 duvdxstep[1]:= (dudx shr 16) + (dvdx shr 16) * TexturePitch; // U+V step
 duvdxstep[0]:= duvdxstep[1] + TexturePitch;                  // + carry

 while (True) do
  begin
   x1:= Ceil(left_x);
   Width:= Ceil(right_x) - x1;

   if (Width > 0) then
    begin
     // Prestep initial texture u,v
     Prestep:= (x1 shl 16) - left_x;
     u:= left_u + iMul16(Prestep, dudx);
     v:= left_v + iMul16(Prestep, dvdx);

     // Calc initial source pointer
     Inner(Pointer(Integer(DestPtr) + x1 * 4), TextureBits, Width, u, v,
      dudx, dvdx, TexturePitch);
    end;

   Inc(Integer(DestPtr), DestPitch);

   // Scan the right side
   Dec(right_height);
   if(right_height <= 0) then // End of this section?
    begin
     repeat
      if (right_vtx = max_vtx) then Exit;
      RightSection();
     until (not (right_height <= 0))
    end else Inc(right_x, right_dxdy);

   // Scan the left side
   Dec(left_height);
   if(left_height <= 0) then // End of this section?
    begin
     repeat
      if (left_vtx = max_vtx) then Exit;
      LeftSection();
     until (not (left_height <= 0));
    end else
    begin
     Inc(left_x, left_dxdy);
     Inc(left_u, left_dudy);
     Inc(left_v, left_dvdy);
    end;
 end;
end;

//--------------------------------------------------------------------------
procedure TexMap(Dest, Texture: TSystemSurface; const v0, v1, v2: TVector3;
 uv0, uv1, uv2: TPoint2);
var
 Vertices: array[0..2] of TVertexUV;
 Denom, id: Single;
 dudx, dvdx: Integer;
begin
 uv0.x:= uv0.x * Texture.Width;
 uv0.y:= uv0.y * Texture.Height;
 uv1.x:= uv1.x * Texture.Width;
 uv1.y:= uv1.y * Texture.Height;
 uv2.x:= uv2.x * Texture.Width;
 uv2.y:= uv2.y * Texture.Height;

 Vertices[0].x:= FloatToFixed(v0.x);
 Vertices[0].y:= FloatToFixed(v0.y);
 Vertices[0].u:= FloatToFixed(uv0.x);
 Vertices[0].v:= FloatToFixed(uv0.y);

 Vertices[1].x:= FloatToFixed(v1.x);
 Vertices[1].y:= FloatToFixed(v1.y);
 Vertices[1].u:= FloatToFixed(uv1.x);
 Vertices[1].v:= FloatToFixed(uv1.y);

 Vertices[2].x:= FloatToFixed(v2.x);
 Vertices[2].y:= FloatToFixed(v2.y);
 Vertices[2].u:= FloatToFixed(uv2.x);
 Vertices[2].v:= FloatToFixed(uv2.y);

 Denom:= (v0.x - v2.x) * (v1.y - v2.y) - (v1.x - v2.x) * (v0.y - v2.y);
 if (Denom = 0.0) then Exit;

 id:= 1.0 / Denom * 65536.0;

 dudx:= Round(((uv0.x - uv2.x) * (v1.y - v2.y) -
  (uv1.x - uv2.x) * (v0.y - v2.y)) * id);
 dvdx:= Round(((uv0.y - uv2.y) * (v1.y - v2.y) -
  (uv1.y - uv2.y) * (v0.y - v2.y)) * id);


 DestBits := Dest.Bits;
 DestPitch:= Dest.Pitch;

 DrawArbitraryTexturePoly(@Vertices[0], 3, Texture.Bits, Texture.Pitch,
  dudx, dvdx);
end;

//---------------------------------------------------------------------------
end.
